# SDOHCCPRAPAREMap - SDOH Clinical Care v3.0.0-ballot

## StructureMap: SDOHCCPRAPAREMap 

 
A StructureMap instance that represents the resource that transforms specific questions and answers from the PRAPARE questionnaire represented in LOINC (code 93025-5) into their corresponding Observations and derived Conditions (health concerns) 

```
/// name = "SDOHCCPRAPAREMap"
/// status = draft
/// title = "SDOHCC StructureMap PRAPARE"
/// description = "A StructureMap instance that represents the resource that transforms specific questions and answers from the PRAPARE questionnaire represented in LOINC (code 93025-5) into their corresponding Observations and derived Conditions (health concerns)."


// This is a more complex example where the author has chosen to extract *all* answers to Observations and to create a grouping Observation

map "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureMap/SDOHCC-StructureMapPRAPARE" = "SDOHCCPRAPAREMap"

uses "http://hl7.org/fhir/uv/sdc/StructureDefinition/sdc-questionnaireresponse" alias questionnaireResponse as source
uses "http://hl7.org/fhir/StructureDefinition/Bundle" as target
uses "http://hl7.org/fhir/StructureDefinition/Observation" as target
uses "http://hl7.org/fhir/StructureDefinition/Condition" alias sdohccCondition as target
uses "http://hl7.org/fhir/StructureDefinition/Observation" alias sdohccObservation as target

group sdohMapping(source src : questionnaireResponse, target bundle : Bundle) {
  //temporarily set id to human-readable id as example
  //src -> bundle.id = uuid();
  src -> bundle.id = 'SDOHCC-BundlePRAPAREExample' "bundleId";
  src -> bundle.type = 'transaction' "bundleType";
    
  //for each answer item in the questionnaireResponse, create an Observation and add it to the groupObservation
  src.item as rootItem then 
    rootContent(src, rootItem, bundle);

  //create group observation containing all observations as hasMember
  src -> bundle.entry as entry, entry.resource = create('sdohccObservation') as groupObservation then 
    TransformGroupObservation(src, bundle, groupObservation, entry) "bundleGroupObsn";  
}

group rootContent(source qResponse: questionnaireResponse, source rootItem, target bundle) {
  rootItem.item as groupItem then 
    groupContent(qResponse, groupItem, bundle);
}

group groupContent(source qResponse: questionnaireResponse, source groupItem, target bundle) {
  groupItem.item as answerItem then
    itemContent(qResponse, answerItem, bundle);
}

group itemContent(source qResponse: questionnaireResponse, source answerItem, target bundle) {
  //create a single Observation for questions that can only have one answer
  answerItem where answerItem.linkId != '/93043-8/32624-9' and answerItem.linkId != '/93041-2/93031-3' and answerItem.linkId != '/93041-2/93030-5'-> bundle.entry as entry, entry.resource = create('sdohccObservation') as observation then 
    TransformObservation(qResponse, answerItem, bundle, observation, entry, bundle) "obsn";

  //create multiple Observations for each answer on questions that can have multiple answers (Question 2, 14 and 15)
  answerItem.answer as answer where answerItem.linkId = '/93043-8/32624-9' or answerItem.linkId = '/93041-2/93031-3' or answerItem.linkId = '/93041-2/93030-5' -> bundle.entry as entry, entry.resource = create('sdohccObservation') as observation then 
    TransformObservation(qResponse, answerItem, bundle, observation, entry, bundle),
    SetValueCodeableConceptFromAnswer(answer, observation);
}

group SetValueCodeableConceptFromAnswer(source answer, target observation) {
  answer -> observation.value = create('CodeableConcept') as newCC then {
    answer.value as coding -> newCC.coding = coding as newCoding;
  } "obsnValue";
}

group TransformObservation(source src: questionnaireResponse, source answerItem, source srcBundle: Bundle, target observation: sdohccObservation, target entry, target bundle: Bundle)
{
  src -> observation.id = uuid() then
  	SetObservationFullUrl(observation, entry) "obsnFullUrl";
  src -> observation.status = 'final' "obsnStatus";
  src -> observation.category = cc('http://terminology.hl7.org/CodeSystem/observation-category', 'social-history', 'Social History') "obsnCatSocialHistory";
  src -> observation.category = cc('http://terminology.hl7.org/CodeSystem/observation-category', 'survey', 'Survey') "obsnCatSurvey";
 
  src -> observation.meta = create('Meta') as newMeta then {
    src -> newMeta.profile = 'http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ObservationScreeningResponse' "obsnMetaProfile";
  } "obsnMeta";

  //set dates
  src.authored as authored -> observation.issued = authored, observation.effective = authored;
  
  src.subject as qSubject -> observation.subject = create('Reference') as newReference then {
    qSubject.reference as qReference -> newReference.reference = qReference;
    qSubject.display as qDisplay -> newReference.display = qDisplay;
  };
  src.subject as qSubject -> observation.performer = create('Reference') as newReference then {
    qSubject.reference as qReference -> newReference.reference = qReference;
    qSubject.display as qDisplay -> newReference.display = qDisplay;
  };  

  src.id as id -> observation.derivedFrom = create('Reference') as newReference then {
    id -> newReference.reference = append('QuestionnaireResponse/', id) "obsnDerivedFromReference";
  } "obsnDerivedFrom";

  //answerItem.answer as answer -> observation.id = uuid() then
  //	SetValue(answer, observation);

  //Set observation.code mapping for each corresponding answer

  // 1. Ethnicity
  answerItem where linkId = '/93043-8/56051-6' -> observation.code = cc('http://loinc.org', '56051-6') as code then 
    SetValueCodeableConcept(answerItem, observation) "obsnCodeEthnicity";
  answerItem where linkId = '/93043-8/56051-6' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 2. Race
  //TBD: How to handle "Others" text?
  answerItem where linkId = '/93043-8/32624-9' -> observation.code = cc('http://loinc.org', '32624-9') as code "obsCodeRace";
  answerItem where linkId = '/93043-8/32624-9' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 3. Farm Worker Status
  answerItem where linkId = '/93043-8/93035-4' -> observation.code = cc('http://loinc.org', '93035-4') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeFarmWorkerStatus";
  answerItem where linkId = '/93043-8/93035-4' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 4. Veteran Status
  answerItem where linkId = '/93043-8/93034-7' -> observation.code = cc('http://loinc.org', '93034-7') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeVetStatus";
  answerItem where linkId = '/93043-8/93034-7' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'veteran-status' "obsnCatCodingCode";
      src -> newCoding.display = 'Veteran Status' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";  
  
  // 5. English Proficiency
  answerItem where linkId = '/93043-8/54899-0' -> observation.code = cc('http://loinc.org', '54899-0') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeEngProf";
  answerItem where linkId = '/93043-8/54899-0' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 6. Household Size
  answerItem where linkId = '/93042-0/63512-8' -> observation.code = cc('http://loinc.org', '63512-8') as code then {
    answerItem.answer as answer -> observation.value = create('Quantity') as newQty then {
      answer.value as vs -> newQty.value = vs "obsnQtyValue";
      src -> newQty.unit = '{#}' "obsnQtyUnit";
      src -> newQty.system = 'http://unitsofmeasure.org' "obsnQtySystem";
      src -> newQty.code = '{#}' "obsnQtyCode";
    } "obsnValue";
  } "obsnCodeHouseholdSize";
  answerItem where linkId = '/93042-0/63512-8' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 7. Housing
  answerItem where linkId = '/93042-0/71802-3' -> observation.code = cc('http://loinc.org', '71802-3') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeHousing";
  answerItem where linkId = '/93042-0/71802-3' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'homelessness' "obsnCatCodingCode";
      src -> newCoding.display = 'Homelessness' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 8. Housing Status
  answerItem where linkId = '/93042-0/93033-9' -> observation.code = cc('http://loinc.org', '93033-9') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeHousingStatus";
  answerItem where linkId = '/93042-0/93033-9' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'housing-instability' "obsnCatCodingCode";
      src -> newCoding.display = 'Housing Instability' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 9. Neighborhood
  answerItem where linkId = '/93042-0/56799-0' -> observation.code = cc('http://loinc.org', '56799-0') as code then
    SetValue(answerItem, observation) "obsnCodeNeighborhood";
  answerItem where linkId = '/93042-0/56799-0' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 10. Education
  answerItem where linkId = '/93041-2/82589-3' -> observation.code = cc('http://loinc.org', '82589-3') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeEducation";
  answerItem where linkId = '/93041-2/82589-3' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'educational-attainment' "obsnCatCodingCode";
      src -> newCoding.display = 'Educational Attainment' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 11. Employment
  answerItem where linkId = '/93041-2/67875-5' -> observation.code = cc('http://loinc.org', '67875-5') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeEmployment";
  answerItem where linkId = '/93041-2/67875-5' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'employment-status' "obsnCatCodingCode";
      src -> newCoding.display = 'Employment Status' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 12. Insurance
  answerItem where linkId = '/93041-2/76437-3' -> observation.code = cc('http://loinc.org', '76437-3') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeInsurance";
  answerItem where linkId = '/93041-2/76437-3' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'health-insurance-coverage-status' "obsnCatCodingCode";
      src -> newCoding.display = 'Health Insurance Coverage Status' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 13. Household Income
  answerItem where linkId = '/93041-2/63586-2' -> observation.code = cc('http://loinc.org', '63586-2') as code then {
    answerItem.answer as answer -> observation.value = create('Quantity') as newQty then {
      answer.value as vs -> newQty.value = vs "obsnQtyValue";
      src -> newQty.unit = '/a' "obsnQtyUnit";
      src -> newQty.system = 'http://unitsofmeasure.org' "obsnQtySystem";
      src -> newQty.code = '/a' "obsnQtyCode";
    } "obsnBalue";
  } "obsnCodeHouseholdIncome";
  answerItem where linkId = '/93041-2/63586-2' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";  

  // 14. Material Security
  answerItem where linkId = '/93041-2/93031-3' -> observation.code = cc('http://loinc.org', '93031-3') as code "obsnCodeMaterialSecurity";
  // answer: Food
  answerItem where linkId = '/93041-2/93031-3' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'material-hardship' "obsnCatCodingCode";
      src -> newCoding.display = 'Material Hardship' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";
  
  // 15. Transportation
  answerItem where linkId = '/93041-2/93030-5' -> observation.code = cc('http://loinc.org', '93030-5') as code "obsnCodeTransportation";
  answerItem where linkId = '/93041-2/93030-5' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'transportation-insecurity' "obsnCatCodingCode";
      src -> newCoding.display = 'Transportation Insecurity' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 16. Social Integration
  answerItem where linkId = '/93040-4/93029-7' -> observation.code = cc('http://loinc.org', '93029-7') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeSocInteg";
  answerItem where linkId = '/93040-4/93029-7' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'social-connection' "obsnCatCodingCode";
      src -> newCoding.display = 'Social Connection' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 17. Stress
  answerItem where linkId = '/93040-4/93038-8' -> observation.code = cc('http://loinc.org', '93038-8') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeStress";
  answerItem where linkId = '/93040-4/93038-8' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'stress' "obsnCatCodingCode";
      src -> newCoding.display = 'Stress' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 18. Incarceration
  answerItem where linkId = '/93039-6/93028-9' -> observation.code = cc('http://loinc.org', '93028-9') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeIncarceration";
  answerItem where linkId = '/93039-6/93028-9' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 19. Refugee
  answerItem where linkId = '/93039-6/93027-1' -> observation.code = cc('http://loinc.org', '93027-1') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeRefugee";
  answerItem where linkId = '/93039-6/93027-1' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 20. Safety
  answerItem where linkId = '/93039-6/93026-3' -> observation.code = cc('http://loinc.org', '93026-3') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeSafety";
  answerItem where linkId = '/93039-6/93026-3' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'sdoh-category-unspecified' "obsnCatCodingCode";
      src -> newCoding.display = 'SDOH Category Unspecified' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  // 21. Domestic Violence
  answerItem where linkId = '/93039-6/76501-6' -> observation.code = cc('http://loinc.org', '76501-6') as code then
    SetValueCodeableConcept(answerItem, observation) "obsnCodeDomViolence";
  answerItem where linkId = '/93039-6/76501-6' -> observation.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "obsnCatCodingSystem";
      src -> newCoding.code = 'intimate-partner-violence' "obsnCatCodingCode";
      src -> newCoding.display = 'Intimate Partner Violence' "obsnCatCodingDisplay";
    } "obsnCatCoding";
  } "obsnCat";

  //create a Condition if homeless
  answerItem where linkId = '/93042-0/71802-3' and answer.value.code ='LA30190-5' -> bundle.entry as entry, entry.resource = create('sdohccCondition') as condition then
    TransformCondition(src, bundle, condition, entry), SetConditionCodeAndEvidence1(src, srcBundle, condition) "conditionHomeless";

  //create a Condition if unemployed
  answerItem where linkId = '/93041-2/67875-5' and answer.value.code ='LA17956-6' -> bundle.entry as entry, entry.resource = create('sdohccCondition') as condition then
    TransformCondition(src, bundle, condition, entry), SetConditionCodeAndEvidence2(src, srcBundle, condition) "conditionUnemployed";

  src -> entry.request as request then {
    src -> request.method = 'POST' "obsnRequestMethod";
    src -> request.url = 'Observation' "obsnRequestUrl";
  } "obsnEntryRequest";
}

group SetValue(source answerItem, target observation) {
  answerItem.answer as answer -> observation.id as id then {
    answer.value as vs -> observation.value = vs;
  };
}

group SetValueCodeableConcept(source src, target observation) {
  src.answer as answer -> observation.value = create('CodeableConcept') as newCC then {
    answer.value as coding -> newCC.coding = coding as newCoding;
  };
}

group SetObservationFullUrl(source observation: Observation, target entry)
{
  observation.id as id -> entry.fullUrl = append('http://hl7.org/fhir/us/sdoh-clinicalcare/Observation/', id);
}

group TransformCondition(source src: questionnaireResponse, source bundle, target condition: sdohccCondition, target entry)
{
  src -> condition.id = uuid() then
    SetConditionFullUrl(condition, entry) "conditionFullUrl";

  src -> condition.meta = create('Meta') as newMeta then {
    src -> newMeta.profile = 'http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition' "conditionMetaProfile";
  } "conditionMeta";

  src -> condition.clinicalStatus = cc('http://terminology.hl7.org/CodeSystem/condition-clinical', 'active', 'Active') "conditionClinicalStatus";
  src -> condition.verificationStatus = cc('http://terminology.hl7.org/CodeSystem/condition-ver-status', 'unconfirmed', 'Unconfirmed') "conditionVerificationStatus";

  src -> condition.category = create('CodeableConcept') as newCC then {
    src -> newCC.coding = create('Coding') as newCoding then {
      src -> newCoding.system = 'http://hl7.org/fhir/us/core/CodeSystem/condition-category' "conditionCatCodingSystem";
      src -> newCoding.code = 'health-concern' "conditionCatCodingCode";
      src -> newCoding.display = 'Health Concern' "conditionCatCodingDisplay";
    } "conditionCatCoding";
  } "conditionCat";
    
  src.authored as authored -> condition.onset = create('Period') as period,
    period.start = authored "conditionOnset";

  src.subject as qSubject -> condition.subject = create('Reference') as newReference then {
    qSubject.reference as qReference -> newReference.reference = qReference;
    qSubject.display as qDisplay -> newReference.display = qDisplay;
  };

  src.subject as qSubject -> condition.asserter = create('Reference') as newReference then {
    qSubject.reference as qReference -> newReference.reference = qReference;
    qSubject.display as qDisplay -> newReference.display = qDisplay;
  };

  src -> entry.request as request then {
    src -> request.method = 'POST' "conditionRequestMethod";
    src -> request.url = 'Condition' "conditionRequestUrl";
  } "conditionEntryRequest";    
}

group SetConditionFullUrl(source condition: sdohccCondition, target entry)
{
  condition.id as id -> entry.fullUrl = append('http://hl7.org/fhir/us/sdoh-clinicalcare/Condition/', id);
}

group SetConditionCodeAndEvidence1(source src: questionnaireResponse, source bundle, target condition)
{
  src -> condition.code = create('CodeableConcept') as newCodeCC then {
      src -> newCodeCC.coding = create('Coding') as newCoding then {
        src -> newCoding.system = 'http://snomed.info/sct' "conditionCodeCodingSystem";
        src -> newCoding.code = '32911000' "conditionCodeCodingCode";
        src -> newCoding.display = 'Homeless' "conditionCodeCodingDisplay";
    } "conditionCoding";
    src -> newCodeCC.coding = create('Coding') as newCoding2 then {
        src -> newCoding2.system = 'http://hl7.org/fhir/sid/icd-10-cm' "conditionCodeCodingSystem";
        src -> newCoding2.code = 'Z59.0' "conditionCodeCodingCode";
        src -> newCoding2.display = 'Homelessness' "conditionCodeCodingDisplay";
    } "conditionCoding";
  } "conditionCode";
  
  //Add sdoh category
  src -> condition.category = create('CodeableConcept') as newCC then {
      src -> newCC.coding = create('Coding') as newCoding then {
        src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "conditionCatCodingSystem";
        src -> newCoding.code = 'homelessness' "conditionCatCodingCode";
        src -> newCoding.display = 'Homelessness' "conditionCatCodingDisplay";
    } "conditionCatCoding";
  } "conditionCat";

  bundle.entry as entries -> condition.evidence as evidence, evidence.detail = create('Reference') as detailReference then {
  	entries.resource as obs where code.coding.code = '71802-3'-> detailReference.reference = reference(obs) "conditionEvidence";
  };
}

group SetConditionCodeAndEvidence2(source src: questionnaireResponse, source bundle, target condition)
{
  src -> condition.code = create('CodeableConcept') as newCodeCC then {
      src -> newCodeCC.coding = create('Coding') as newCoding then {
        src -> newCoding.system = 'http://snomed.info/sct' "conditionCodeCodingSystem";
        src -> newCoding.code = '73438004' "conditionCodeCodingCode";
        src -> newCoding.display = 'Unemployed' "conditionCodeCodingDisplay";
    } "conditionCoding";
    src -> newCodeCC.coding = create('Coding') as newCoding2 then {
        src -> newCoding2.system = 'http://hl7.org/fhir/sid/icd-10-cm' "conditionCodeCodingSystem";
        src -> newCoding2.code = 'Z56.0' "conditionCodeCodingCode";
        src -> newCoding2.display = 'Unemployment, unspecified' "conditionCodeCodingDisplay";
    } "conditionCoding";
  } "conditionCode"; 
  //Add sdoh category
  src -> condition.category = create('CodeableConcept') as newCC then {
      src -> newCC.coding = create('Coding') as newCoding then {
        src -> newCoding.system = 'http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes' "conditionCatCodingSystem";
        src -> newCoding.code = 'employment-status' "conditionCatCodingCode";
        src -> newCoding.display = 'Employment Status' "conditionCatCodingDisplay";
    } "conditionCatCoding";
  }"conditionCat";  
  bundle.entry as entries -> condition.evidence as evidence, evidence.detail = create('Reference') as detailReference then {
  	entries.resource as obs where code.coding.code = '67875-5'-> detailReference.reference = reference(obs)  "conditionEvidence"; 
  };
}

group TransformGroupObservation(source src: questionnaireResponse, source bundle:Bundle, target groupObservation: Observation, target entry)
{
  src -> groupObservation.id = uuid() then
    SetObservationFullUrl(groupObservation, entry) "grpObsnFullUrl"; 
  src -> groupObservation.status = 'final' "grpObsnStatus"; 
  src -> groupObservation.category = cc('http://terminology.hl7.org/CodeSystem/observation-category', 'social-history', 'Social History') "grpObsnCatSocialHistory";
  src -> groupObservation.category = cc('http://terminology.hl7.org/CodeSystem/observation-category', 'survey', 'Survey') "grpObsnCatSurvey";

  src -> groupObservation.meta = create('Meta') as newMeta then {
    src -> newMeta.profile = 'http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ObservationScreeningResponse' "grpObsnMetaProfile";
  } "grpObsMeta"; 

  src.authored as authored -> groupObservation.effective = authored;
  src.subject as qSubject -> groupObservation.subject = create('Reference') as newReference then {
    qSubject.reference as qReference -> newReference.reference = qReference "grpObsnSubjRef";
    qSubject.display as qDisplay -> newReference.display = qDisplay "grpObsnSubjDisplay";
  } "grpObsnSubject";  

  src -> groupObservation.code = cc('http://loinc.org', '93025-5') "grpObsnCode";   

  //only add observations that are derived from the answers
  bundle.entry as entries -> groupObservation.hasMember = create('Reference') as df then {
    entries.resource as obs where status.exists() and hasMember.exists().not() -> df.reference = reference(obs) "grpObsnHasMemberReference"; 
  } "grpObsnHasMember";

  src.id as id -> groupObservation.derivedFrom = create('Reference') as newReference then {
    id -> newReference.reference = append('QuestionnaireResponse/', id) "grpObsnDerivedFromReference";
  } "grpObsnDerivedFrom";

  src -> entry.request as request then {
    src -> request.method = 'POST' "obsnRequestMethod";
    src -> request.url = 'Observation' "obsnRequestUrl";
  } "obsnEntryRequest";    
}
```



## Resource Content

```json
{
  "resourceType" : "StructureMap",
  "id" : "SDOHCC-StructureMapPRAPARE",
  "extension" : [
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/structuredefinition-wg",
      "valueCode" : "pc"
    }
  ],
  "url" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureMap/SDOHCC-StructureMapPRAPARE",
  "version" : "3.0.0-ballot",
  "name" : "SDOHCCPRAPAREMap",
  "status" : "draft",
  "date" : "2025-12-13T17:11:59+00:00",
  "publisher" : "HL7 International / Patient Care",
  "contact" : [
    {
      "name" : "HL7 International / Patient Care",
      "telecom" : [
        {
          "system" : "url",
          "value" : "http://www.hl7.org/Special/committees/patientcare"
        },
        {
          "system" : "email",
          "value" : "patientcare@lists.hl7.org"
        }
      ]
    }
  ],
  "description" : "A StructureMap instance that represents the resource that transforms specific questions and answers from the PRAPARE questionnaire represented in LOINC (code 93025-5) into their corresponding Observations and derived Conditions (health concerns)",
  "jurisdiction" : [
    {
      "coding" : [
        {
          "system" : "urn:iso:std:iso:3166",
          "code" : "US",
          "display" : "United States of America"
        }
      ]
    }
  ],
  "structure" : [
    {
      "url" : "http://hl7.org/fhir/uv/sdc/StructureDefinition/sdc-questionnaireresponse",
      "mode" : "source",
      "alias" : "questionnaireResponse"
    },
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/Bundle",
      "mode" : "target"
    },
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/Observation",
      "mode" : "target"
    },
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/Condition",
      "mode" : "target",
      "alias" : "sdohccCondition"
    },
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/Observation",
      "mode" : "target",
      "alias" : "sdohccObservation"
    }
  ],
  "group" : [
    {
      "name" : "sdohMapping",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "src",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "type" : "Bundle",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "bundleId",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "bundle",
              "contextType" : "variable",
              "element" : "id",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueString" : "SDOHCC-BundlePRAPAREExample"
                }
              ]
            }
          ]
        },
        {
          "name" : "bundleType",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "bundle",
              "contextType" : "variable",
              "element" : "type",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueString" : "transaction"
                }
              ]
            }
          ]
        },
        {
          "name" : "item",
          "source" : [
            {
              "context" : "src",
              "element" : "item",
              "variable" : "rootItem"
            }
          ],
          "dependent" : [
            {
              "name" : "rootContent",
              "variable" : ["src", "rootItem", "bundle"]
            }
          ]
        },
        {
          "name" : "bundleGroupObsn",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "bundle",
              "contextType" : "variable",
              "element" : "entry",
              "variable" : "entry"
            },
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "resource",
              "variable" : "groupObservation",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "sdohccObservation"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "TransformGroupObservation",
              "variable" : ["src", "bundle", "groupObservation", "entry"]
            }
          ]
        }
      ]
    },
    {
      "name" : "rootContent",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "qResponse",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "rootItem",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "item",
          "source" : [
            {
              "context" : "rootItem",
              "element" : "item",
              "variable" : "groupItem"
            }
          ],
          "dependent" : [
            {
              "name" : "groupContent",
              "variable" : ["qResponse", "groupItem", "bundle"]
            }
          ]
        }
      ]
    },
    {
      "name" : "groupContent",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "qResponse",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "groupItem",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "item",
          "source" : [
            {
              "context" : "groupItem",
              "element" : "item",
              "variable" : "answerItem"
            }
          ],
          "dependent" : [
            {
              "name" : "itemContent",
              "variable" : ["qResponse", "answerItem", "bundle"]
            }
          ]
        }
      ]
    },
    {
      "name" : "itemContent",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "qResponse",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "answerItem",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "obsn",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "(answerItem.linkId != '/93043-8/32624-9') and (answerItem.linkId != '/93041-2/93031-3') and (answerItem.linkId != '/93041-2/93030-5')"
            }
          ],
          "target" : [
            {
              "context" : "bundle",
              "contextType" : "variable",
              "element" : "entry",
              "variable" : "entry"
            },
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "resource",
              "variable" : "observation",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "sdohccObservation"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "TransformObservation",
              "variable" : ["qResponse", "answerItem", "bundle", "observation", "entry", "bundle"]
            }
          ]
        },
        {
          "name" : "answer",
          "source" : [
            {
              "context" : "answerItem",
              "element" : "answer",
              "variable" : "answer",
              "condition" : "(answerItem.linkId = '/93043-8/32624-9') or (answerItem.linkId = '/93041-2/93031-3') or (answerItem.linkId = '/93041-2/93030-5')"
            }
          ],
          "target" : [
            {
              "context" : "bundle",
              "contextType" : "variable",
              "element" : "entry",
              "variable" : "entry"
            },
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "resource",
              "variable" : "observation",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "sdohccObservation"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "TransformObservation",
              "variable" : ["qResponse", "answerItem", "bundle", "observation", "entry", "bundle"]
            },
            {
              "name" : "SetValueCodeableConceptFromAnswer",
              "variable" : ["answer", "observation"]
            }
          ]
        }
      ]
    },
    {
      "name" : "SetValueCodeableConceptFromAnswer",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "answer",
          "mode" : "source"
        },
        {
          "name" : "observation",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "obsnValue",
          "source" : [
            {
              "context" : "answer",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "value",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "value",
              "source" : [
                {
                  "context" : "answer",
                  "element" : "value",
                  "variable" : "coding"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "coding"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "TransformObservation",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "src",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "answerItem",
          "mode" : "source"
        },
        {
          "name" : "srcBundle",
          "type" : "Bundle",
          "mode" : "source"
        },
        {
          "name" : "observation",
          "type" : "sdohccObservation",
          "mode" : "target"
        },
        {
          "name" : "entry",
          "mode" : "target"
        },
        {
          "name" : "bundle",
          "type" : "Bundle",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "obsnFullUrl",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "id",
              "transform" : "uuid"
            }
          ],
          "dependent" : [
            {
              "name" : "SetObservationFullUrl",
              "variable" : ["observation", "entry"]
            }
          ]
        },
        {
          "name" : "obsnStatus",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "status",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueString" : "final"
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCatSocialHistory",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://terminology.hl7.org/CodeSystem/observation-category"
                },
                {
                  "valueString" : "social-history"
                },
                {
                  "valueString" : "Social History"
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCatSurvey",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://terminology.hl7.org/CodeSystem/observation-category"
                },
                {
                  "valueString" : "survey"
                },
                {
                  "valueString" : "Survey"
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnMeta",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "meta",
              "variable" : "newMeta",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Meta"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnMetaProfile",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newMeta",
                  "contextType" : "variable",
                  "element" : "profile",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ObservationScreeningResponse"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "authored",
          "source" : [
            {
              "context" : "src",
              "element" : "authored",
              "variable" : "authored"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "issued",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueId" : "authored"
                }
              ]
            },
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "effective",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueId" : "authored"
                }
              ]
            }
          ]
        },
        {
          "name" : "subject",
          "source" : [
            {
              "context" : "src",
              "element" : "subject",
              "variable" : "qSubject"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "subject",
              "variable" : "newReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "reference",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "reference",
                  "variable" : "qReference"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qReference"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "display",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "display",
                  "variable" : "qDisplay"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "display",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qDisplay"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "subject",
          "source" : [
            {
              "context" : "src",
              "element" : "subject",
              "variable" : "qSubject"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "performer",
              "variable" : "newReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "reference",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "reference",
                  "variable" : "qReference"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qReference"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "display",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "display",
                  "variable" : "qDisplay"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "display",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qDisplay"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnDerivedFrom",
          "source" : [
            {
              "context" : "src",
              "element" : "id",
              "variable" : "id"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "derivedFrom",
              "variable" : "newReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnDerivedFromReference",
              "source" : [
                {
                  "context" : "id",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "append",
                  "parameter" : [
                    {
                      "valueString" : "QuestionnaireResponse/"
                    },
                    {
                      "valueId" : "id"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeEthnicity",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/56051-6'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "56051-6"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/56051-6'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsCodeRace",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/32624-9'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "32624-9"
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/32624-9'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeFarmWorkerStatus",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/93035-4'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93035-4"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/93035-4'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeVetStatus",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/93034-7'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93034-7"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/93034-7'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "veteran-status"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Veteran Status"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeEngProf",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/54899-0'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "54899-0"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93043-8/54899-0'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeHouseholdSize",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/63512-8'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "63512-8"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnValue",
              "source" : [
                {
                  "context" : "answerItem",
                  "element" : "answer",
                  "variable" : "answer"
                }
              ],
              "target" : [
                {
                  "context" : "observation",
                  "contextType" : "variable",
                  "element" : "value",
                  "variable" : "newQty",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Quantity"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnQtyValue",
                  "source" : [
                    {
                      "context" : "answer",
                      "element" : "value",
                      "variable" : "vs"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "value",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueId" : "vs"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnQtyUnit",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "unit",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "{#}"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnQtySystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://unitsofmeasure.org"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnQtyCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "{#}"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/63512-8'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeHousing",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/71802-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "71802-3"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/71802-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "homelessness"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Homelessness"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeHousingStatus",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/93033-9'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93033-9"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/93033-9'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "housing-instability"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Housing Instability"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeNeighborhood",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/56799-0'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "56799-0"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValue",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93042-0/56799-0'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeEducation",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/82589-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "82589-3"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/82589-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "educational-attainment"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Educational Attainment"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeEmployment",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/67875-5'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "67875-5"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/67875-5'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "employment-status"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Employment Status"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeInsurance",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/76437-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "76437-3"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/76437-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "health-insurance-coverage-status"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Health Insurance Coverage Status"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeHouseholdIncome",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/63586-2'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "63586-2"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnBalue",
              "source" : [
                {
                  "context" : "answerItem",
                  "element" : "answer",
                  "variable" : "answer"
                }
              ],
              "target" : [
                {
                  "context" : "observation",
                  "contextType" : "variable",
                  "element" : "value",
                  "variable" : "newQty",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Quantity"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnQtyValue",
                  "source" : [
                    {
                      "context" : "answer",
                      "element" : "value",
                      "variable" : "vs"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "value",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueId" : "vs"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnQtyUnit",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "unit",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "/a"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnQtySystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://unitsofmeasure.org"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnQtyCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newQty",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "/a"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/63586-2'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeMaterialSecurity",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/93031-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93031-3"
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/93031-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "material-hardship"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Material Hardship"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeTransportation",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/93030-5'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93030-5"
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93041-2/93030-5'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "transportation-insecurity"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Transportation Insecurity"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeSocInteg",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93040-4/93029-7'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93029-7"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93040-4/93029-7'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "social-connection"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Social Connection"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeStress",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93040-4/93038-8'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93038-8"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93040-4/93038-8'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "stress"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Stress"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeIncarceration",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/93028-9'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93028-9"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/93028-9'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeRefugee",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/93027-1'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93027-1"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/93027-1'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeSafety",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/93026-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93026-3"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/93026-3'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "sdoh-category-unspecified"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "SDOH Category Unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnCodeDomViolence",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/76501-6'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "76501-6"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "SetValueCodeableConcept",
              "variable" : ["answerItem", "observation"]
            }
          ]
        },
        {
          "name" : "obsnCat",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "linkId = '/93039-6/76501-6'"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "obsnCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "obsnCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "intimate-partner-violence"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "obsnCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Intimate Partner Violence"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionHomeless",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "(linkId = '/93042-0/71802-3') and (answer.value.code = 'LA30190-5')"
            }
          ],
          "target" : [
            {
              "context" : "bundle",
              "contextType" : "variable",
              "element" : "entry",
              "variable" : "entry"
            },
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "resource",
              "variable" : "condition",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "sdohccCondition"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "TransformCondition",
              "variable" : ["src", "bundle", "condition", "entry"]
            },
            {
              "name" : "SetConditionCodeAndEvidence1",
              "variable" : ["src", "srcBundle", "condition"]
            }
          ]
        },
        {
          "name" : "conditionUnemployed",
          "source" : [
            {
              "context" : "answerItem",
              "variable" : "variable",
              "condition" : "(linkId = '/93041-2/67875-5') and (answer.value.code = 'LA17956-6')"
            }
          ],
          "target" : [
            {
              "context" : "bundle",
              "contextType" : "variable",
              "element" : "entry",
              "variable" : "entry"
            },
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "resource",
              "variable" : "condition",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "sdohccCondition"
                }
              ]
            }
          ],
          "dependent" : [
            {
              "name" : "TransformCondition",
              "variable" : ["src", "bundle", "condition", "entry"]
            },
            {
              "name" : "SetConditionCodeAndEvidence2",
              "variable" : ["src", "srcBundle", "condition"]
            }
          ]
        },
        {
          "name" : "obsnEntryRequest",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "request",
              "variable" : "request"
            }
          ],
          "rule" : [
            {
              "name" : "obsnRequestMethod",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "request",
                  "contextType" : "variable",
                  "element" : "method",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "POST"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "obsnRequestUrl",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "request",
                  "contextType" : "variable",
                  "element" : "url",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "Observation"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "SetValue",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "answerItem",
          "mode" : "source"
        },
        {
          "name" : "observation",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "answer",
          "source" : [
            {
              "context" : "answerItem",
              "element" : "answer",
              "variable" : "answer"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "id",
              "variable" : "id"
            }
          ],
          "rule" : [
            {
              "name" : "value",
              "source" : [
                {
                  "context" : "answer",
                  "element" : "value",
                  "variable" : "vs"
                }
              ],
              "target" : [
                {
                  "context" : "observation",
                  "contextType" : "variable",
                  "element" : "value",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "vs"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "SetValueCodeableConcept",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "src",
          "mode" : "source"
        },
        {
          "name" : "observation",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "answer",
          "source" : [
            {
              "context" : "src",
              "element" : "answer",
              "variable" : "answer"
            }
          ],
          "target" : [
            {
              "context" : "observation",
              "contextType" : "variable",
              "element" : "value",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "value",
              "source" : [
                {
                  "context" : "answer",
                  "element" : "value",
                  "variable" : "coding"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "coding"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "SetObservationFullUrl",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "observation",
          "type" : "Observation",
          "mode" : "source"
        },
        {
          "name" : "entry",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "id",
          "source" : [
            {
              "context" : "observation",
              "element" : "id",
              "variable" : "id"
            }
          ],
          "target" : [
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "fullUrl",
              "transform" : "append",
              "parameter" : [
                {
                  "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/Observation/"
                },
                {
                  "valueId" : "id"
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "TransformCondition",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "src",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "mode" : "source"
        },
        {
          "name" : "condition",
          "type" : "sdohccCondition",
          "mode" : "target"
        },
        {
          "name" : "entry",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "conditionFullUrl",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "id",
              "transform" : "uuid"
            }
          ],
          "dependent" : [
            {
              "name" : "SetConditionFullUrl",
              "variable" : ["condition", "entry"]
            }
          ]
        },
        {
          "name" : "conditionMeta",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "meta",
              "variable" : "newMeta",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Meta"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionMetaProfile",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newMeta",
                  "contextType" : "variable",
                  "element" : "profile",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionClinicalStatus",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "clinicalStatus",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://terminology.hl7.org/CodeSystem/condition-clinical"
                },
                {
                  "valueString" : "active"
                },
                {
                  "valueString" : "Active"
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionVerificationStatus",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "verificationStatus",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://terminology.hl7.org/CodeSystem/condition-ver-status"
                },
                {
                  "valueString" : "unconfirmed"
                },
                {
                  "valueString" : "Unconfirmed"
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionCat",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "conditionCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/core/CodeSystem/condition-category"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "health-concern"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Health Concern"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionOnset",
          "source" : [
            {
              "context" : "src",
              "element" : "authored",
              "variable" : "authored"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "onset",
              "variable" : "period",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Period"
                }
              ]
            },
            {
              "context" : "period",
              "contextType" : "variable",
              "element" : "start",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueId" : "authored"
                }
              ]
            }
          ]
        },
        {
          "name" : "subject",
          "source" : [
            {
              "context" : "src",
              "element" : "subject",
              "variable" : "qSubject"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "subject",
              "variable" : "newReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "reference",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "reference",
                  "variable" : "qReference"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qReference"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "display",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "display",
                  "variable" : "qDisplay"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "display",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qDisplay"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "subject",
          "source" : [
            {
              "context" : "src",
              "element" : "subject",
              "variable" : "qSubject"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "asserter",
              "variable" : "newReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "reference",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "reference",
                  "variable" : "qReference"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qReference"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "display",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "display",
                  "variable" : "qDisplay"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "display",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qDisplay"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionEntryRequest",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "request",
              "variable" : "request"
            }
          ],
          "rule" : [
            {
              "name" : "conditionRequestMethod",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "request",
                  "contextType" : "variable",
                  "element" : "method",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "POST"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "conditionRequestUrl",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "request",
                  "contextType" : "variable",
                  "element" : "url",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "Condition"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "SetConditionFullUrl",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "condition",
          "type" : "sdohccCondition",
          "mode" : "source"
        },
        {
          "name" : "entry",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "id",
          "source" : [
            {
              "context" : "condition",
              "element" : "id",
              "variable" : "id"
            }
          ],
          "target" : [
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "fullUrl",
              "transform" : "append",
              "parameter" : [
                {
                  "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/Condition/"
                },
                {
                  "valueId" : "id"
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "SetConditionCodeAndEvidence1",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "src",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "mode" : "source"
        },
        {
          "name" : "condition",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "conditionCode",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "newCodeCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCodeCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "conditionCodeCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://snomed.info/sct"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "32911000"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Homeless"
                        }
                      ]
                    }
                  ]
                }
              ]
            },
            {
              "name" : "conditionCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCodeCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding2",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "conditionCodeCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding2",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/sid/icd-10-cm"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding2",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Z59.0"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding2",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Homelessness"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionCat",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "conditionCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "homelessness"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Homelessness"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "entry",
          "source" : [
            {
              "context" : "bundle",
              "element" : "entry",
              "variable" : "entries"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "evidence",
              "variable" : "evidence"
            },
            {
              "context" : "evidence",
              "contextType" : "variable",
              "element" : "detail",
              "variable" : "detailReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionEvidence",
              "source" : [
                {
                  "context" : "entries",
                  "element" : "resource",
                  "variable" : "obs",
                  "condition" : "code.coding.code = '71802-3'"
                }
              ],
              "target" : [
                {
                  "context" : "detailReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "reference",
                  "parameter" : [
                    {
                      "valueId" : "obs"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "SetConditionCodeAndEvidence2",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "src",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "mode" : "source"
        },
        {
          "name" : "condition",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "conditionCode",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "code",
              "variable" : "newCodeCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCodeCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "conditionCodeCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://snomed.info/sct"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "73438004"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Unemployed"
                        }
                      ]
                    }
                  ]
                }
              ]
            },
            {
              "name" : "conditionCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCodeCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding2",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "conditionCodeCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding2",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/sid/icd-10-cm"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding2",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Z56.0"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCodeCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding2",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Unemployment, unspecified"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "conditionCat",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "category",
              "variable" : "newCC",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "CodeableConcept"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionCatCoding",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newCC",
                  "contextType" : "variable",
                  "element" : "coding",
                  "variable" : "newCoding",
                  "transform" : "create",
                  "parameter" : [
                    {
                      "valueString" : "Coding"
                    }
                  ]
                }
              ],
              "rule" : [
                {
                  "name" : "conditionCatCodingSystem",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "system",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCatCodingCode",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "code",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "employment-status"
                        }
                      ]
                    }
                  ]
                },
                {
                  "name" : "conditionCatCodingDisplay",
                  "source" : [
                    {
                      "context" : "src",
                      "variable" : "src"
                    }
                  ],
                  "target" : [
                    {
                      "context" : "newCoding",
                      "contextType" : "variable",
                      "element" : "display",
                      "transform" : "copy",
                      "parameter" : [
                        {
                          "valueString" : "Employment Status"
                        }
                      ]
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "entry",
          "source" : [
            {
              "context" : "bundle",
              "element" : "entry",
              "variable" : "entries"
            }
          ],
          "target" : [
            {
              "context" : "condition",
              "contextType" : "variable",
              "element" : "evidence",
              "variable" : "evidence"
            },
            {
              "context" : "evidence",
              "contextType" : "variable",
              "element" : "detail",
              "variable" : "detailReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "conditionEvidence",
              "source" : [
                {
                  "context" : "entries",
                  "element" : "resource",
                  "variable" : "obs",
                  "condition" : "code.coding.code = '67875-5'"
                }
              ],
              "target" : [
                {
                  "context" : "detailReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "reference",
                  "parameter" : [
                    {
                      "valueId" : "obs"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    },
    {
      "name" : "TransformGroupObservation",
      "typeMode" : "none",
      "input" : [
        {
          "name" : "src",
          "type" : "questionnaireResponse",
          "mode" : "source"
        },
        {
          "name" : "bundle",
          "type" : "Bundle",
          "mode" : "source"
        },
        {
          "name" : "groupObservation",
          "type" : "Observation",
          "mode" : "target"
        },
        {
          "name" : "entry",
          "mode" : "target"
        }
      ],
      "rule" : [
        {
          "name" : "grpObsnFullUrl",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "id",
              "transform" : "uuid"
            }
          ],
          "dependent" : [
            {
              "name" : "SetObservationFullUrl",
              "variable" : ["groupObservation", "entry"]
            }
          ]
        },
        {
          "name" : "grpObsnStatus",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "status",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueString" : "final"
                }
              ]
            }
          ]
        },
        {
          "name" : "grpObsnCatSocialHistory",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "category",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://terminology.hl7.org/CodeSystem/observation-category"
                },
                {
                  "valueString" : "social-history"
                },
                {
                  "valueString" : "Social History"
                }
              ]
            }
          ]
        },
        {
          "name" : "grpObsnCatSurvey",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "category",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://terminology.hl7.org/CodeSystem/observation-category"
                },
                {
                  "valueString" : "survey"
                },
                {
                  "valueString" : "Survey"
                }
              ]
            }
          ]
        },
        {
          "name" : "grpObsMeta",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "meta",
              "variable" : "newMeta",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Meta"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "grpObsnMetaProfile",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newMeta",
                  "contextType" : "variable",
                  "element" : "profile",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ObservationScreeningResponse"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "authored",
          "source" : [
            {
              "context" : "src",
              "element" : "authored",
              "variable" : "authored"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "effective",
              "transform" : "copy",
              "parameter" : [
                {
                  "valueId" : "authored"
                }
              ]
            }
          ]
        },
        {
          "name" : "grpObsnSubject",
          "source" : [
            {
              "context" : "src",
              "element" : "subject",
              "variable" : "qSubject"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "subject",
              "variable" : "newReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "grpObsnSubjRef",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "reference",
                  "variable" : "qReference"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qReference"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "grpObsnSubjDisplay",
              "source" : [
                {
                  "context" : "qSubject",
                  "element" : "display",
                  "variable" : "qDisplay"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "display",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueId" : "qDisplay"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "grpObsnCode",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "code",
              "transform" : "cc",
              "parameter" : [
                {
                  "valueString" : "http://loinc.org"
                },
                {
                  "valueString" : "93025-5"
                }
              ]
            }
          ]
        },
        {
          "name" : "grpObsnHasMember",
          "source" : [
            {
              "context" : "bundle",
              "element" : "entry",
              "variable" : "entries"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "hasMember",
              "variable" : "df",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "grpObsnHasMemberReference",
              "source" : [
                {
                  "context" : "entries",
                  "element" : "resource",
                  "variable" : "obs",
                  "condition" : "status.exists() and hasMember.exists().not()"
                }
              ],
              "target" : [
                {
                  "context" : "df",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "reference",
                  "parameter" : [
                    {
                      "valueId" : "obs"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "grpObsnDerivedFrom",
          "source" : [
            {
              "context" : "src",
              "element" : "id",
              "variable" : "id"
            }
          ],
          "target" : [
            {
              "context" : "groupObservation",
              "contextType" : "variable",
              "element" : "derivedFrom",
              "variable" : "newReference",
              "transform" : "create",
              "parameter" : [
                {
                  "valueString" : "Reference"
                }
              ]
            }
          ],
          "rule" : [
            {
              "name" : "grpObsnDerivedFromReference",
              "source" : [
                {
                  "context" : "id",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "newReference",
                  "contextType" : "variable",
                  "element" : "reference",
                  "transform" : "append",
                  "parameter" : [
                    {
                      "valueString" : "QuestionnaireResponse/"
                    },
                    {
                      "valueId" : "id"
                    }
                  ]
                }
              ]
            }
          ]
        },
        {
          "name" : "obsnEntryRequest",
          "source" : [
            {
              "context" : "src",
              "variable" : "src"
            }
          ],
          "target" : [
            {
              "context" : "entry",
              "contextType" : "variable",
              "element" : "request",
              "variable" : "request"
            }
          ],
          "rule" : [
            {
              "name" : "obsnRequestMethod",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "request",
                  "contextType" : "variable",
                  "element" : "method",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "POST"
                    }
                  ]
                }
              ]
            },
            {
              "name" : "obsnRequestUrl",
              "source" : [
                {
                  "context" : "src",
                  "variable" : "src"
                }
              ],
              "target" : [
                {
                  "context" : "request",
                  "contextType" : "variable",
                  "element" : "url",
                  "transform" : "copy",
                  "parameter" : [
                    {
                      "valueString" : "Observation"
                    }
                  ]
                }
              ]
            }
          ]
        }
      ]
    }
  ]
}

```
