# SDOHCC Condition - SDOH Clinical Care v3.0.0-ballot

## Resource Profile: SDOHCC Condition 

 
Profile for Social Determinants of Health (SDOH) conditions. 

The Gravity Project team updated the SDOHCC Condition profile to include Protective Factors as a possible SDOHCC domain. We would encourage balloters to review this guidance and provide feedback.

### Purpose

SDOHCC Condition is generally intended to represent a health concern, problem, diagnosis related to a social risk or need, or a protective factor. SDOHCC Condition may be the result of a screening response Observation (e.g., [SDOHCC Observation Screening Response](StructureDefinition-SDOHCC-ObservationScreeningResponse.md) or other Observations) made as part of an encounter assessment (e.g., [SDOHCC Observation Assessment](StructureDefinition-SDOHCC-ObservationAssessment.md)). SDOHCC Condition may be addressed by a Goal ([SDOHCC Goal](StructureDefinition-SDOHCC-Goal.md)) or be the reason for a ServiceRequest ([SDOHCC Service Request](StructureDefinition-SDOHCC-ServiceRequest.md)) or Procedure ([SDOHCC Procedure](StructureDefinition-SDOHCC-Procedure.md)). See the [Data Modeling Framework](sdoh_clinical_care_scope.md#data-modeling-framework) for more detail on the relationships between SDOHCC Condition and the other profiles in this IG.

### Usage

`Condition.code` has a binding to [US Core Condition Codes](http://hl7.org/fhir/us/core/STU7/ValueSet-us-core-condition-code.html) ([required](http://hl7.org/fhir/R4/terminologies.html#required)). `Condition.code` also has additional bindings to SDOH domain-specific value sets (e.g., for Food Insecurity) which are determined by the value selected for `Condition.category`. When `Condition.category` is a value from [SDOHCC ValueSet SDOH Category](ValueSet-SDOHCC-ValueSetSDOHCategory.md) (e.g., food-insecurity), `Condition.code` **SHALL** be selected from the corresponding domain-specific value set found in Additional Bindings (i.e. [Food Insecurity Diagnoses](https://vsac.nlm.nih.gov/valueset/2.16.840.1.113762.1.4.1247.17/expansion)). If more than one value is selected from [SDOHCC ValueSet SDOH Category](ValueSet-SDOHCC-ValueSetSDOHCategory.md) (e.g., food-insecurity **and** financial-insecurity) for `Condition.category`, then `Condition.code` **SHALL** be selected from at least one of the corresponding domain-specific value sets found in Additional Bindings (e.g. [Food Insecurity Diagnoses](https://vsac.nlm.nih.gov/valueset/2.16.840.1.113762.1.4.1247.17/expansion)).

Of note, Gravity Project is continuously developing terminology for new SDOH domains. Sometimes, a new SDOH domain may already be represented by a category code in [SDOHCC ValueSet SDOH Category](ValueSet-SDOHCC-ValueSetSDOHCategory.md), but a domain-specific value set may not yet be available in VSAC. In this case, the Additional Binding for the new SDOH domain/category is the default binding for the profile element. Additionally, sometimes a domain-specific value set for a new domain is available in VSAC but does not yet include all standardized codes identified during the domain’s terminology development cycle. In this case, the value set’s “Description” tab in VSAC will state, “In progress”. Additional information related to Gravity value sets is provided on the [Social Risk Terminology Value Sets](https://confluence.hl7.org/display/GRAV/Social+Risk+Terminology+Value+Sets) page on the Gravity Project Confluence site.

Additionally, some SDOH domains may not yet be represented in [SDOHCC ValueSet SDOH Category](ValueSet-SDOHCC-ValueSetSDOHCategory.md). In this case, `Condition.category`: [sdoh-category-unspecified](CodeSystem-SDOHCC-CodeSystemTemporaryCodes.md#SDOHCC-CodeSystemTemporaryCodes-sdoh-category-unspecified) can be used so that the Condition instance can still be categorized as SDOH-related. If desired, `Condition.category.text` can also be used to further specify the domain. For example, at the time of publication of this IG, “disability-status” is not yet included in [SDOHCC ValueSet SDOH Category](ValueSet-SDOHCC-ValueSetSDOHCategory.md). In this case, `Condition.category`: [sdoh-category-unspecified](CodeSystem-SDOHCC-CodeSystemTemporaryCodes.md#SDOHCC-CodeSystemTemporaryCodes-sdoh-category-unspecified) can be used along with `Condition.category.text`: Disability Status. For `Condition.category`: [sdoh-category-unspecified](CodeSystem-SDOHCC-CodeSystemTemporaryCodes.md#SDOHCC-CodeSystemTemporaryCodes-sdoh-category-unspecified), there will not be domain-specific value sets and Additional Bindings for `Condition.code` will designate the default binding to [US Core Condition Codes](http://hl7.org/fhir/us/core/STU7/ValueSet-us-core-condition-code.html) ([required](http://hl7.org/fhir/R4/terminologies.html#required)).

For more information on the correlation between `Condition.category` and the domain-specific value sets in Additional Bindings for `Condition.code`, see [Gravity Project Value Sets](gravity_terminology.md#gravity-project-value-sets).

### US Core Conformance

US Core 6.1.0 and beyond profiles two Condition profiles: Encounter Diagnosis Condition Profile and Problems and Health Concerns Condition profile. The primary difference between these two profiles is in the `Condition.category` data element, where each profile requires different values to determine the category of the Condition resource. Instances derived from the SDOHCC Condition profile will be conformant to one of the US Core Condition profiles. `Condition.category` **SHALL** be one of encounter-diagnosis, problem-list-item, or health-concern.

### Protective Factors

In addition to representing social risks or needs, the SDOHCC Condition profile is also used to represent protective factors. Protective factors are defined as characteristics or strengths of individuals, families, or communities that act to mitigate risks and promote positive well-being. To represent a protective factor using this profile, `Condition.category` **SHALL** include the value `protective-factor`, in addition to adhering to all other constraints on the `Condition.category` element. Consistent with the binding logic for other SDOH domains, when `Condition.category` includes `protective-factor`, the `Condition.code` **SHALL** be selected from the corresponding domain-specific value set (e.g., Protective Factor Diagnoses) found in Additional Bindings. Examples of protective factors that can be represented include findings like "Spiritual strength" , "Stably housed" , or "Psychological resilience".

**Usages:**

* Refer to this Profile: [SDOHCC Condition](StructureDefinition-SDOHCC-Condition.md), [SDOHCC Goal](StructureDefinition-SDOHCC-Goal.md), [SDOHCC Procedure](StructureDefinition-SDOHCC-Procedure.md), [SDOHCC ServiceRequest](StructureDefinition-SDOHCC-ServiceRequest.md) and [SDOHCC Task For Referral Management](StructureDefinition-SDOHCC-TaskForReferralManagement.md)
* Examples for this Profile: [Condition/SDOHCC-ConditionAsserterDeviceExample](Condition-SDOHCC-ConditionAsserterDeviceExample.md), [Condition/SDOHCC-ConditionFoodInsecurityExample](Condition-SDOHCC-ConditionFoodInsecurityExample.md), [Condition/SDOHCC-ConditionHomelessExample](Condition-SDOHCC-ConditionHomelessExample.md), [Condition/SDOHCC-ConditionProtectiveFactor](Condition-SDOHCC-ConditionProtectiveFactor.md) and [Condition/SDOHCC-ConditionUnemployedExample](Condition-SDOHCC-ConditionUnemployedExample.md)
* CapabilityStatements using this Profile: [SDOHCC Coordination Platform](CapabilityStatement-SDOHCC-CoordinationPlatform.md), [SDOHCC Patient Application](CapabilityStatement-SDOHCC-PatientApp.md), [SDOHCC Referral Recipient](CapabilityStatement-SDOHCC-ReferralRecipient.md), [SDOHCC Referral Recipient - Light](CapabilityStatement-SDOHCC-ReferralRecipientLight.md) and [SDOHCC Referral Source](CapabilityStatement-SDOHCC-ReferralSource.md)

You can also check for [usages in the FHIR IG Statistics](https://packages2.fhir.org/xig/hl7.fhir.us.sdoh-clinicalcare|current/StructureDefinition/SDOHCC-Condition)

### Formal Views of Profile Content

 [Description Differentials, Snapshots, and other representations](http://build.fhir.org/ig/FHIR/ig-guidance/readingIgs.html#structure-definitions). 

 

Other representations of profile: [CSV](../StructureDefinition-SDOHCC-Condition.csv), [Excel](../StructureDefinition-SDOHCC-Condition.xlsx), [Schematron](../StructureDefinition-SDOHCC-Condition.sch) 



## Resource Content

```json
{
  "resourceType" : "StructureDefinition",
  "id" : "SDOHCC-Condition",
  "extension" : [
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/structuredefinition-wg",
      "valueCode" : "pc"
    },
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/structuredefinition-fmm",
      "valueInteger" : 2,
      "_valueInteger" : {
        "extension" : [
          {
            "url" : "http://hl7.org/fhir/StructureDefinition/structuredefinition-conformance-derivedFrom",
            "valueCanonical" : "http://hl7.org/fhir/us/sdoh-clinicalcare/ImplementationGuide/hl7.fhir.us.sdoh-clinicalcare"
          }
        ]
      }
    },
    {
      "url" : "http://hl7.org/fhir/StructureDefinition/structuredefinition-standards-status",
      "valueCode" : "trial-use",
      "_valueCode" : {
        "extension" : [
          {
            "url" : "http://hl7.org/fhir/StructureDefinition/structuredefinition-conformance-derivedFrom",
            "valueCanonical" : "http://hl7.org/fhir/us/sdoh-clinicalcare/ImplementationGuide/hl7.fhir.us.sdoh-clinicalcare"
          }
        ]
      }
    }
  ],
  "url" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
  "version" : "3.0.0-ballot",
  "name" : "SDOHCCCondition",
  "title" : "SDOHCC Condition",
  "status" : "active",
  "date" : "2025-12-13T17:11:59+00:00",
  "publisher" : "HL7 International / Patient Care",
  "contact" : [
    {
      "name" : "HL7 International / Patient Care",
      "telecom" : [
        {
          "system" : "url",
          "value" : "http://www.hl7.org/Special/committees/patientcare"
        },
        {
          "system" : "email",
          "value" : "patientcare@lists.hl7.org"
        }
      ]
    }
  ],
  "description" : "Profile for Social Determinants of Health (SDOH) conditions.",
  "jurisdiction" : [
    {
      "coding" : [
        {
          "system" : "urn:iso:std:iso:3166",
          "code" : "US",
          "display" : "United States of America"
        }
      ]
    }
  ],
  "fhirVersion" : "4.0.1",
  "mapping" : [
    {
      "identity" : "workflow",
      "uri" : "http://hl7.org/fhir/workflow",
      "name" : "Workflow Pattern"
    },
    {
      "identity" : "sct-concept",
      "uri" : "http://snomed.info/conceptdomain",
      "name" : "SNOMED CT Concept Domain Binding"
    },
    {
      "identity" : "v2",
      "uri" : "http://hl7.org/v2",
      "name" : "HL7 v2 Mapping"
    },
    {
      "identity" : "rim",
      "uri" : "http://hl7.org/v3",
      "name" : "RIM Mapping"
    },
    {
      "identity" : "w5",
      "uri" : "http://hl7.org/fhir/fivews",
      "name" : "FiveWs Pattern Mapping"
    },
    {
      "identity" : "sct-attr",
      "uri" : "http://snomed.org/attributebinding",
      "name" : "SNOMED CT Attribute Binding"
    }
  ],
  "kind" : "resource",
  "abstract" : false,
  "type" : "Condition",
  "baseDefinition" : "http://hl7.org/fhir/StructureDefinition/Condition",
  "derivation" : "constraint",
  "differential" : {
    "element" : [
      {
        "id" : "Condition",
        "path" : "Condition",
        "short" : "Detailed information about SDOH conditions, problems or diagnoses",
        "definition" : "For representation of SDOH conditions.",
        "comment" : "Many of the SDOHCC profiles reference one another. One flow supported by this IG is that QuestionnaireResponses result in Observations that can be used as evidence for Conditions that can lead to Goals, ServiceRequests and Procedures. However, alternatives paths (e.g., to arrive at SDOH Conditions) are also possible.\r\n\r\nOne specific use for this profile is to represent a Health Concern that is either; 1) directly asserted based on the patient’s answer to a specific question from an SDOH screening questionnaire or 2) computed/generated based on the patient’s answers to multiple questions. The questions and answers from the screening questionnaire are also represented using SDC Questionnaire, SDC QuestionnaireResponse and SDOHCC Screening Response Observation.",
        "constraint" : [
          {
            "key" : "SDOH-Con-1",
            "severity" : "error",
            "human" : "Can only have a max of one asserter.",
            "expression" : "extension('http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ExtensionConditionAsserter').count()+asserter.count()<=1",
            "xpath" : "true()",
            "source" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition"
          },
          {
            "key" : "SDOH-Con-2",
            "severity" : "error",
            "human" : "At least one Condition.category SHALL be encounter-diagnosis, problem-list-item, or health-concern",
            "expression" : "category.where(coding.where(system = 'http://terminology.hl7.org/CodeSystem/condition-category' and code='encounter-diagnosis').exists()).exists() or category.where(coding.memberOf('http://hl7.org/fhir/us/core/ValueSet/us-core-problem-or-health-concern')).exists()",
            "xpath" : "true()",
            "source" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition"
          },
          {
            "key" : "SDOH-Con-3",
            "severity" : "error",
            "human" : "At least one Condition.category SHALL be sdoh",
            "expression" : "category.where(coding.where(system = 'http://hl7.org/fhir/us/core/CodeSystem/us-core-category' and code='sdoh').exists()).exists()",
            "source" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition"
          }
        ]
      },
      {
        "id" : "Condition.meta",
        "path" : "Condition.meta",
        "mustSupport" : true
      },
      {
        "id" : "Condition.meta.lastUpdated",
        "path" : "Condition.meta.lastUpdated",
        "short" : "When the resource last changed",
        "mustSupport" : true
      },
      {
        "id" : "Condition.extension",
        "path" : "Condition.extension",
        "slicing" : {
          "discriminator" : [
            {
              "type" : "value",
              "path" : "url"
            }
          ],
          "rules" : "open"
        }
      },
      {
        "id" : "Condition.extension:asserterDevice",
        "path" : "Condition.extension",
        "sliceName" : "asserterDevice",
        "min" : 0,
        "max" : "1",
        "type" : [
          {
            "code" : "Extension",
            "profile" : [
              "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ExtensionConditionAsserter"
            ]
          }
        ],
        "condition" : ["SDOH-Con-1"],
        "mustSupport" : true
      },
      {
        "id" : "Condition.extension:assertedDate",
        "path" : "Condition.extension",
        "sliceName" : "assertedDate",
        "short" : "Date the condition was first asserted",
        "min" : 0,
        "max" : "1",
        "type" : [
          {
            "code" : "Extension",
            "profile" : [
              "http://hl7.org/fhir/StructureDefinition/condition-assertedDate|5.2.0"
            ]
          }
        ],
        "mustSupport" : true
      },
      {
        "id" : "Condition.clinicalStatus",
        "path" : "Condition.clinicalStatus",
        "mustSupport" : true,
        "binding" : {
          "strength" : "required",
          "valueSet" : "http://hl7.org/fhir/ValueSet/condition-clinical"
        }
      },
      {
        "id" : "Condition.verificationStatus",
        "path" : "Condition.verificationStatus",
        "comment" : "verificationStatus is not required.  For example, when a patient complains of stress during a provider encounter, there is not likely going to be a verification status. The data type is CodeableConcept because verificationStatus has some clinical judgment involved, such that there might need to be more specificity than the required FHIR value set allows. For example, a SNOMED coding might allow for additional specificity.\r\n\r\nFor SDOH conditions that are autogenerated based on a questionnaire response, the Condition.asserter is a “Device” AND Condition.Category should be \"health-concern\". In that case, Condition.verificationStatus should be \"unconfirmed\" and should be changed to “confirmed” only after review and concurrence by the provider and patient.",
        "mustSupport" : true,
        "binding" : {
          "strength" : "required",
          "valueSet" : "http://hl7.org/fhir/ValueSet/condition-ver-status"
        }
      },
      {
        "id" : "Condition.category",
        "path" : "Condition.category",
        "slicing" : {
          "discriminator" : [
            {
              "type" : "value",
              "path" : "$this"
            }
          ],
          "rules" : "open"
        },
        "short" : "category codes",
        "min" : 2,
        "condition" : ["SDOH-Con-2"],
        "mustSupport" : true
      },
      {
        "id" : "Condition.category:encounter-diagnosis",
        "path" : "Condition.category",
        "sliceName" : "encounter-diagnosis",
        "short" : "encounter-diagnosis",
        "min" : 0,
        "max" : "1",
        "patternCodeableConcept" : {
          "coding" : [
            {
              "system" : "http://terminology.hl7.org/CodeSystem/condition-category",
              "code" : "encounter-diagnosis"
            }
          ]
        },
        "mustSupport" : true
      },
      {
        "id" : "Condition.category:problem-or-health-concern",
        "path" : "Condition.category",
        "sliceName" : "problem-or-health-concern",
        "short" : "problem-list-item | health-concern",
        "min" : 0,
        "max" : "*",
        "mustSupport" : true,
        "binding" : {
          "strength" : "required",
          "valueSet" : "http://hl7.org/fhir/us/core/ValueSet/us-core-problem-or-health-concern|7.0.0"
        }
      },
      {
        "id" : "Condition.category:screening-assessment",
        "path" : "Condition.category",
        "sliceName" : "screening-assessment",
        "short" : "USCDI Health Status/Assessments Data Class. Requires at least a category code 'sdoh'",
        "definition" : "Categories that a provider may use in their workflow to classify that this Condition is related to a USCDI Health Status/Assessments Data Class.",
        "requirements" : "Used for filtering condition",
        "min" : 1,
        "max" : "*",
        "condition" : ["SDOH-Con-3"],
        "mustSupport" : true,
        "binding" : {
          "strength" : "required",
          "valueSet" : "http://hl7.org/fhir/us/core/ValueSet/us-core-screening-assessment-condition-category|7.0.0"
        }
      },
      {
        "id" : "Condition.category:SDOHCC",
        "path" : "Condition.category",
        "sliceName" : "SDOHCC",
        "short" : "e.g., food-insecurity | transportation-insecurity",
        "definition" : "An SDOH category assigned to the condition.",
        "requirements" : "Codes from this value set can be used to assign one or more SDOH categories (e.g., food-insecurity, transportation-insecurity, etc.) to a condition. It is recommended that SDOH category codes be used to facilitate searching for SDOH conditions.",
        "min" : 0,
        "max" : "*",
        "mustSupport" : true,
        "binding" : {
          "strength" : "required",
          "description" : "Codes for high-level SDOH categories.",
          "valueSet" : "http://hl7.org/fhir/us/sdoh-clinicalcare/ValueSet/SDOHCC-ValueSetSDOHCategory"
        }
      },
      {
        "id" : "Condition.code",
        "path" : "Condition.code",
        "requirements" : "Code is required and must be selected from the bound value set.",
        "min" : 1,
        "mustSupport" : true,
        "binding" : {
          "extension" : [
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "digital-access"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.231"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "digital-access"
                        }
                      ],
                      "text" : "digital-access"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "digital-literacy"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.224"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "digital-literacy"
                        }
                      ],
                      "text" : "digital-literacy"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "educational-attainment"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.103|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "educational-attainment"
                        }
                      ],
                      "text" : "educational-attainment"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "elder-abuse"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.63"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "elder-abuse"
                        }
                      ],
                      "text" : "elder-abuse"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "employment-status"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.42|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "employment-status"
                        }
                      ],
                      "text" : "employment-status"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "financial-insecurity"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.108|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "financial-insecurity"
                        }
                      ],
                      "text" : "financial-insecurity"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "food-insecurity"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.17|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "food-insecurity"
                        }
                      ],
                      "text" : "food-insecurity"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "health-insurance-coverage-status"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.148|20241214"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "health-insurance-coverage-status"
                        }
                      ],
                      "text" : "health-insurance-coverage-status"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "homelessness"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.18|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "homelessness"
                        }
                      ],
                      "text" : "homelessness"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "housing-instability"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.24|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "housing-instability"
                        }
                      ],
                      "text" : "housing-instability"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "inadequate-housing"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.48|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "inadequate-housing"
                        }
                      ],
                      "text" : "inadequate-housing"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "incarceration-status"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.258"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "incarceration-status"
                        }
                      ],
                      "text" : "incarceration-status"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "intimate-partner-violence"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.84|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "intimate-partner-violence"
                        }
                      ],
                      "text" : "intimate-partner-violence"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "language-access"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.262"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "language-access"
                        }
                      ],
                      "text" : "language-access"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "material-hardship"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.35|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "material-hardship"
                        }
                      ],
                      "text" : "material-hardship"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "medical-cost-burden"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.153|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "medical-cost-burden"
                        }
                      ],
                      "text" : "medical-cost-burden"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "personal-health-literacy"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.116|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "personal-health-literacy"
                        }
                      ],
                      "text" : "personal-health-literacy"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "social-connection"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.81|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "social-connection"
                        }
                      ],
                      "text" : "social-connection"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "stress"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.75|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "stress"
                        }
                      ],
                      "text" : "stress"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "transportation-insecurity"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.26|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "transportation-insecurity"
                        }
                      ],
                      "text" : "transportation-insecurity"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "utility-insecurity"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.243"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "utility-insecurity"
                        }
                      ],
                      "text" : "utility-insecurity"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "veteran-status"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.78|20240604"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "veteran-status"
                        }
                      ],
                      "text" : "veteran-status"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "protective-factor"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "extensible"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://cts.nlm.nih.gov/fhir/ValueSet/2.16.840.1.113762.1.4.1247.311"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "protective-factor"
                        }
                      ],
                      "text" : "protective-factor"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            },
            {
              "extension" : [
                {
                  "url" : "key",
                  "valueId" : "sdoh-category-unspecified"
                },
                {
                  "url" : "purpose",
                  "valueCode" : "required"
                },
                {
                  "url" : "valueSet",
                  "valueCanonical" : "http://hl7.org/fhir/us/core/ValueSet/us-core-condition-code|7.0.0"
                },
                {
                  "url" : "usage",
                  "valueUsageContext" : {
                    "code" : {
                      "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
                      "code" : "Condition.category"
                    },
                    "valueCodeableConcept" : {
                      "coding" : [
                        {
                          "system" : "http://hl7.org/fhir/us/sdoh-clinicalcare/CodeSystem/SDOHCC-CodeSystemTemporaryCodes",
                          "code" : "sdoh-category-unspecified"
                        }
                      ],
                      "text" : "sdoh-category-unspecified"
                    }
                  }
                }
              ],
              "url" : "http://hl7.org/fhir/tools/StructureDefinition/additional-binding"
            }
          ],
          "strength" : "required",
          "description" : "Valueset to describe the actual problem experienced by the patient",
          "valueSet" : "http://hl7.org/fhir/us/core/ValueSet/us-core-condition-code|7.0.0"
        }
      },
      {
        "id" : "Condition.bodySite",
        "path" : "Condition.bodySite",
        "max" : "0"
      },
      {
        "id" : "Condition.subject",
        "path" : "Condition.subject",
        "type" : [
          {
            "code" : "Reference",
            "targetProfile" : [
              "http://hl7.org/fhir/us/core/StructureDefinition/us-core-patient|7.0.0"
            ]
          }
        ],
        "mustSupport" : true
      },
      {
        "id" : "Condition.encounter",
        "path" : "Condition.encounter",
        "type" : [
          {
            "code" : "Reference",
            "targetProfile" : [
              "http://hl7.org/fhir/us/core/StructureDefinition/us-core-encounter|7.0.0"
            ]
          }
        ],
        "mustSupport" : true
      },
      {
        "id" : "Condition.onset[x]",
        "path" : "Condition.onset[x]",
        "type" : [
          {
            "extension" : [
              {
                "url" : "http://hl7.org/fhir/StructureDefinition/elementdefinition-type-must-support",
                "valueBoolean" : true
              }
            ],
            "code" : "dateTime"
          },
          {
            "code" : "Period"
          }
        ],
        "mustSupport" : true
      },
      {
        "id" : "Condition.abatement[x]",
        "path" : "Condition.abatement[x]",
        "definition" : "The estimated or actual dateTime or Period that the condition resolved or went into remission. This is called \"abatement\" because of the many overloaded connotations associated with \"remission\" or \"resolution\".",
        "comment" : "There is no explicit distinction between resolution and remission because in many cases the distinction is not clear. If there is no abatement element, it is unknown whether the condition has resolved or entered remission; applications and users should generally assume that the condition is still valid.  When abatementString exists, it implies the condition is abated.\r\n\r\nFor SDOH Conditions that have a fuzzy abatement period, a lower precision representation (e.g., month/year or year) may be used. However, for SDOH Conditions that end at a specific point in time (e.g., food insecurity may abate upon acquiring a new job or gaining eligibility to a food program) a higher precision representation (e.g., year/month/date) may also be used.",
        "type" : [
          {
            "extension" : [
              {
                "url" : "http://hl7.org/fhir/StructureDefinition/elementdefinition-type-must-support",
                "valueBoolean" : true
              }
            ],
            "code" : "dateTime"
          },
          {
            "code" : "Period"
          }
        ],
        "mustSupport" : true
      },
      {
        "id" : "Condition.recordedDate",
        "path" : "Condition.recordedDate",
        "mustSupport" : true
      },
      {
        "id" : "Condition.recorder",
        "path" : "Condition.recorder",
        "type" : [
          {
            "code" : "Reference",
            "targetProfile" : [
              "http://hl7.org/fhir/StructureDefinition/Practitioner",
              "http://hl7.org/fhir/StructureDefinition/PractitionerRole"
            ]
          }
        ]
      },
      {
        "id" : "Condition.asserter",
        "path" : "Condition.asserter",
        "short" : "Person or device that asserts this condition",
        "definition" : "The individual or device making the condition statement.",
        "comment" : "References SHALL be a reference to an actual FHIR resource, and SHALL be resolveable (allowing for access control, temporary unavailability, etc.). Resolution can be either by retrieval from the URL, or, where applicable by resource type, by treating an absolute reference as a canonical URL and looking it up in a local registry/repository.\r\n\r\nFor a condition that is autogenerated based on the patient directly answering a specific question from an SDOH screening questionnaire, the Condition.asserter should be the patient. For a condition that is autogenerated based on a computation that uses the patient’s answers to multiple questions, Condition.asserter should reference “Device” (using the using AsserterDevice extension) to specify the device used to compute the condition.",
        "type" : [
          {
            "code" : "Reference",
            "targetProfile" : [
              "http://hl7.org/fhir/StructureDefinition/RelatedPerson",
              "http://hl7.org/fhir/us/core/StructureDefinition/us-core-patient|7.0.0",
              "http://hl7.org/fhir/us/core/StructureDefinition/us-core-practitioner|7.0.0",
              "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-practitionerrole"
            ]
          }
        ],
        "condition" : ["SDOH-Con-1"],
        "mustSupport" : true
      },
      {
        "id" : "Condition.stage",
        "path" : "Condition.stage",
        "max" : "0"
      },
      {
        "id" : "Condition.evidence",
        "path" : "Condition.evidence",
        "mustSupport" : true
      },
      {
        "id" : "Condition.evidence.code",
        "path" : "Condition.evidence.code",
        "comment" : "Not all terminology uses fit this general pattern. In some cases, models should not use CodeableConcept and use Coding directly and provide their own structure for managing text, codings, translations and the relationship between elements and pre- and post-coordination.\r\n\r\nInformation represented by Condition.evidence.code may overlap significantly with information represented by Condition.evidence.detail. Multiple approaches to representing the same information may negatively impact interoperability. Therefore, where similar infomation could be provided by either Condition.evidence.code or Condition.evidence.detail, it is recommended that Condition.evidence.detail should be used as supporting evidence for the condition.\r\n\r"
      },
      {
        "id" : "Condition.evidence.detail",
        "path" : "Condition.evidence.detail",
        "slicing" : {
          "discriminator" : [
            {
              "type" : "profile",
              "path" : "resolve()"
            }
          ],
          "rules" : "open"
        },
        "comment" : "References SHALL be a reference to an actual FHIR resource, and SHALL be resolveable (allowing for access control, temporary unavailability, etc.). Resolution can be either by retrieval from the URL, or, where applicable by resource type, by treating an absolute reference as a canonical URL and looking it up in a local registry/repository.\r\n\r\nAdditionally, see Comment for Condition.evidence.code."
      },
      {
        "id" : "Condition.evidence.detail:SupportedDetail",
        "path" : "Condition.evidence.detail",
        "sliceName" : "SupportedDetail",
        "definition" : "Links to other relevant information.",
        "requirements" : "When a condition is based on one or more SDOH observations that were generated from questionnaire responses, Condition.evidence.detail should reference instances that comply with the SDOHCC Observation Screening Response or SDC Questionnaire Response. However, references to other instance types are also possible.",
        "min" : 0,
        "max" : "*",
        "type" : [
          {
            "code" : "Reference",
            "targetProfile" : [
              "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ObservationScreeningResponse",
              "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-Condition",
              "http://hl7.org/fhir/us/sdoh-clinicalcare/StructureDefinition/SDOHCC-ObservationAssessment",
              "http://hl7.org/fhir/uv/sdc/StructureDefinition/sdc-questionnaireresponse"
            ]
          }
        ],
        "mustSupport" : true
      }
    ]
  }
}

```
